from fastapi import FastAPI, Depends, HTTPException
from database import Base, engine, SessionLocal
from models import User
from schemas import UserCreate, UserResponse, UserUpdate, AdminCreate, LoginSchema
from fastapi import status
from utils import hash_password, verify_password, create_token
from auth import get_db, get_current_user, get_admin_user
from fastapi.security import OAuth2PasswordRequestForm
from typing import List
from sqlalchemy.orm import Session
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from fastapi.responses import FileResponse
import os

app = FastAPI()
Base.metadata.create_all(bind=engine)

app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:3000"],  # React dev server
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

app.mount("/static", StaticFiles(directory="build/static"), name="static")

@app.get("/", include_in_schema=False)
def serve_react_index():
    return FileResponse("build/index.html")
    
@app.get("/{full_path:path}", include_in_schema=False)
def serve_react_app(full_path: str):
    file_path = f"build/{full_path}"
    if os.path.exists(file_path) and not os.path.isdir(file_path):
        return FileResponse(file_path)
    return FileResponse("frontend/build/index.html")

@app.post("/signup", response_model=UserResponse)
def signup(data: UserCreate, db=Depends(get_db)):
    if db.query(User).filter(User.email == data.email).first():
        raise HTTPException(status_code=400, detail="Email already registered")
    new_user = User(
        fname=data.fname,
        lname=data.lname,
        email=data.email,
        mobile=data.mobile,
        password=hash_password(data.password),
    )
    db.add(new_user)
    db.commit()
    db.refresh(new_user)
    return new_user

@app.post("/login")
def login(data: LoginSchema, db: Session = Depends(get_db)):
    user = db.query(User).filter(User.email == data.email).first()

    if not user or not verify_password(data.password, user.password):
        raise HTTPException(
            status_code=401,
            detail="Invalid email or password"
        )

    if not user.is_approved:
        raise HTTPException(
            status_code=403,
            detail="Your account is pending approval by admin."
        )

    token = create_token({"sub": str(user.id)})

    return {
        "access_token": token,
        "token_type": "bearer",
        "user": {
            "id": user.id,
            "email": user.email,
            "name": f"{user.fname} {user.lname}",
            "role": user.role
        }
    }

@app.get("/admin/users", response_model=List[UserResponse])
def get_users(db=Depends(get_db), admin=Depends(get_admin_user)):
    return db.query(User).filter(User.role == "USER").all()


@app.patch("/admin/approve/{user_id}")
def approve_user(user_id: int, db=Depends(get_db), admin=Depends(get_admin_user)):
    user = db.query(User).filter(User.id == user_id).first()
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    user.is_approved = True
    db.commit()
    return {"message": "User approved"}

@app.patch("/admin/disapprove/{user_id}")
def disapprove_user(user_id: int, db=Depends(get_db), admin=Depends(get_admin_user)):
    user = db.query(User).filter(User.id == user_id).first()
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    user.is_approved = False
    db.commit()
    return {"message": "User disapproved"}

@app.patch("/admin/users/{user_id}", response_model=UserResponse)
def update_user(user_id: int, data: UserUpdate, db=Depends(get_db), admin=Depends(get_admin_user)):
    user = db.query(User).filter(User.id == user_id).first()
    if not user:
        raise HTTPException(status_code=404, detail="User not found")

    # Update only provided fields
    if data.fname is not None:
        user.fname = data.fname
    if data.lname is not None:
        user.lname = data.lname
    if data.mobile is not None:
        existing = db.query(User).filter(User.mobile == data.mobile, User.id != user_id).first()
        if existing:
            raise HTTPException(status_code=400, detail="Mobile already in use")
        user.mobile = data.mobile

    db.commit()
    db.refresh(user)
    return user

@app.delete("/admin/users/{user_id}")
def delete_user(user_id: int, db=Depends(get_db), admin=Depends(get_admin_user)):
    user = db.query(User).filter(User.id == user_id).first()
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    db.delete(user)
    db.commit()
    return {"message": "User deleted"}

ADMIN_SECRET = "create-admin-2024"  # change this to a secure secret

@app.post("/admin/signup", response_model=UserResponse, status_code=status.HTTP_201_CREATED)
def admin_signup(data: AdminCreate, db=Depends(get_db)):
    if data.secret_key != ADMIN_SECRET:
        raise HTTPException(status_code=403, detail="Invalid secret key")
    
    existing_user = db.query(User).filter(User.email == data.email).first()
    if existing_user:
        raise HTTPException(status_code=400, detail="Email already registered")

    admin_user = User(
        fname=data.fname,
        lname=data.lname,
        email=data.email,
        mobile=data.mobile,
        password=hash_password(data.password),
        role="ADMIN",
        is_approved=True
    )
    db.add(admin_user)
    db.commit()
    db.refresh(admin_user)
    return admin_user